#include <string.h>
#include <bastard.h>

char * base_isa_str[] = {
"nop", "add", "and", "b", "blcr", "bset", "btog", "btst", "call", "clr", 
"cmp", "dec", "div", "inc", "jmp", "ld", "mod", "mul", "mv", "neg",
"not", "or", "restore", "ret", "rol", "save", "set", "sll", "sra", "srl",
"st", "sub", "swap", "t", "tret", "test", "xor"
};

char * ext_isa_str[] = {
".nop", ".label", ".data", ".global", ".frame", ".unframe", ".proc", ".asm", 
".block", ".unblock", ".calc", ".uncalc", ".clobber"
}; 

char * cc_str[]  = {
	"",		/* empty */
      "n",  "a",  /*  NEVER, ALWAYS  */
      "e",  "ne", /*  EQUAL, NOT EQUAL */
      "g",  "le", /*  GREATER, LESSER/EQUAL */
      "l",  "ge", /*  LESSER, GREATER/EQUAL */
      "neg","pos",      /*  NEGATIVE, POSITIVE */
      "cs", "cc", /*  CARRY, NO CARRY */
      "vs", "vc"  /*  OVERFLOW, NO OVERFLOW */
};

char * sz_str[] = {
	"",			/* none */
      "b", "ub",      	/* 1, 2 */    /* 00001b == signed */
      "h", "uh",    	/* 3, 4 */
      "w", "uw",      	/* 5, 6 */
      "d", "ud",    	/* 7, 8 */
      "q", "uq",    	/* 9, A */
      "x"            	/* B */    /* extended precision */
};

/* NOTE: use '%' to prefix registers */
char * reg_fmt_str[] = {
	"", "%s%%g%d", "%s%%i%d", "%s%%o%d", "%s%%l%d", "%s%%r%d"
};

char * spec_reg_str[] = {
	"%sp", "%fp", "%pc", "%cz", "%cn", "%cv", "%cc"
};

/* NOTE: use '$' to prefix immediate values */
char * sz_fmt_str[] = {
	"%s$%+d", "%s$0x%02X", "%s$%+d", "%s$0x%X", "%s$%+ld", "%s$0x%lX"
};

char * intcode_getstr_cc( int x ) {
	if ( x < 0 || x > 14 ) return( cc_str[0] );
	return( cc_str[x] );
}

char * intcode_getstr_sz( int x ) {
	if ( x < 0 || x > 11 ) return( sz_str[0] );
	return( sz_str[x] );
}

char * intcode_getstr_sreg( int reg ) {
	if ( reg < 0 || reg > 6 ) return( spec_reg_str[0] );
	return( spec_reg_str[reg] );
}

int intcode_getstr_mnem( unsigned long opcode, char *buf, int len ) {
	int type, inst, x;

	type = opcode >> 24;
	inst = opcode >> 16 & 0xFF;

	if (! type ) {
		if ( inst < 0 || inst > i_xor ) inst = 0;
		strncat( buf, base_isa_str[inst], len - strlen(buf));
	} else if ( type == 0x10 ) {
		if ( inst < 0 || inst > i_clobber ) inst = 0;
		strncat( buf, ext_isa_str[inst], len - strlen(buf));
	}

	/* do opcode suffixes */
	if ( (x = opcode & 0xF00) ) {
		strncat( buf, intcode_getstr_cc( x >> 8 ), len - strlen(buf) );
	}
	if ( (x = opcode & 0xF0) ) {
		strncat( buf, intcode_getstr_sz( x >> 4 ), len - strlen(buf) );
	}
	if ( (x = opcode & 0x0F) ) {
		strncat( buf, intcode_getstr_sz( x ), len - strlen(buf) );
	}
	return( strlen(buf) );
}


int intcode_sprint_op(unsigned long op, unsigned long type, char *buf, int len){
	struct name n = {0};
	int size;

	memset( buf, 0, len );
	len -= 2;		/* one for the NULL, one for a '*' if needed */
	
	if ( type & DEREF_OP ) strcat( buf, "*" );
	switch ( type & 0xFFFF ) {
		case op_greg: case op_ireg: case op_oreg: 
		case op_lreg: case op_rreg:
			snprintf(buf, len, reg_fmt_str[type & 0xFFFF], buf, op); 	
			break;
		case op_specreg:
			strncat(buf, intcode_getstr_sreg( op ), len );
			break;
		case op_label: 
			if (! bdb_index_find( NAME_ID, &op, &n ) ) {
				snprintf( buf, len, "%s?(%08X)", buf, op );
			} else 
				strncat( buf, n.text, len );
			break;
		case op_dir:
			/* these are context-specific... */
			break;
		case op_imm: 
		default:
			/* yeah, this needs work */
			size = type & 0xF0000 >> 16;
			if ( size < 1 || size > 6 ) size = 1;
			snprintf(buf, len, sz_fmt_str[size], buf, op); 	
			break;
	}
	return(1);
}

int intcode_sprint_dirop( struct int_code *i, char *buf, int len ) {
	struct comment cmt = {0};
	struct code c = {0};
	struct function f = {0};
	struct func_local l = {0};
	struct name n = {0};
	char opbuf[64] = {0};

	/* do any directive operands */
	switch ( i->opcode >> 16 & 0xFF ) {
		case i_label:
			/* src is RVA of a (NAMEd) CODE object */
		case i_global:
			/* src is (NAMEd) ADDRESS object */
			if ( bdb_index_find( NAME_RVA, &i->src, &n ) ) 
				strncat(buf, n.text, len - strlen(buf));
			else {
				sprintf(opbuf, "unk_%08X", buf, i->src);
				strncat(buf, opbuf, len - strlen(buf));
			}
			break;
		case i_data:
			/* src is FUNC_LOCAL */
			if ( bdb_index_find( FUNC_LOCAL_ID, &i->src, &l ) ) { 
				strncat(buf, l.name, len - strlen(buf));
			}
			break;
		case i_proc:
			/* src is FUNCTION object */
			if ( bdb_index_find( FUNCTION_ID, &i->src, &f ) ) { 
				if ( bdb_index_find( NAME_RVA, &f.rva, &n ) ) 
					strncat(buf, n.text, len - strlen(buf));
			}
			break;
		case i_asm:
			/* src is CODE object */
			code_sprintf(opbuf, "%m\t%s%,%d%,%t", i->src);
			//snprintf( buf, len - strlen(buf), "%s\"%s\"", buf, opbuf);
			strcat( buf, "\"" );
			strncat( buf, opbuf, len - strlen(buf) - 1 );
			strcat( buf, "\"" );
			break;
		case i_clobber:
			/* register number */
			intcode_sprint_op( i->src, i->sType, opbuf, 32 );
			strncat( buf, opbuf, len - strlen(buf) );
			break;
		case i_calc:
			addrexp_get_str(i->src, opbuf, 128);
			strcat( buf, "\"" );
			strncat( buf, opbuf, len - strlen(buf) - 1 );
			strcat( buf, "\"" );
			break;
		case i_block:
			/* src is INT_CODE id of condition owning this block */
			//sprintf( buf, "%s(id %x)", buf, i->id); break;
		case i_unblock:
			/* src is INT_CODE id of .block */
			//sprintf( buf, "%s(id %x)", buf, i->src); break;
		case i_uncalc:
		case i_frame:
		case i_unframe:
		default:
			/* no operand */
			break;
	}
	return(1);
}

int intcode_sprint( struct int_code *i, char *buf, int len ) {
	int x, need_comma = 0;
	char opbuf[64] = {0};
	struct comment c = {0};
	
	memset( buf, 0, len);	
	len--;	/* subtract space for NULL */


	/* tab this puppy */
	if ( ! i->opcode & EXT_ISA )
	strcat( buf, "\t" );

	/* do main opcode */
	intcode_getstr_mnem( i->opcode, buf, len );

	/* do operands */
	if ( i->opcode & EXT_ISA ) { /* handle differently if directive */
		intcode_sprint_dirop( i, opbuf, 64 );
		strcat( buf, "\t" );
		strncat( buf, opbuf, len - strlen(buf) );
	} else {
		/* do src */
		if ( i->sType ) {
			intcode_sprint_op( i->src, i->sType, opbuf, 64 );
			strcat( buf, "\t" );
			strncat( buf, opbuf, len - strlen(buf) );
			need_comma = 1;
		}
		/* do arg */
		if ( i->aType ) {
			intcode_sprint_op( i->arg, i->aType, opbuf, 64 );
			if ( need_comma ) 
				strncat( buf, ", ", len - strlen(buf) -1 );
			strcat( buf, "\t" );
			strncat( buf, opbuf, len - strlen(buf) );
			need_comma = 1;
		}
		/* do dest */
		if ( i->dType ) {
			intcode_sprint_op( i->dest, i->dType, opbuf, 64 );
			if ( need_comma ) 
				strncat( buf, ", ", len - strlen(buf) -1 );
			strcat( buf, "\t" );
			strncat( buf, opbuf, len - strlen(buf) - 1 );
		}
	}

	if ( i->comment ) {
		strcat( buf, "\t #" );
		bdb_index_find( COMMENT_ID, &i->comment, &c );
		strncat( buf, c.text, len - strlen(buf) );
	}
	
	return(1);
}

int intcode_gen( struct function *f, int options ) {
	if (! f) return(0);
	ext_arch->options |= options;
	return( ext_gen_int(f) );
}

int intcode_sprintf(char *buf, char *format, struct int_code *i)
{
/* This takes a printf-style format string and uses it to
 * format an intermediate code object to a buffer. Format
 * specifiers are:
 *    %m - mnemonic          %s - src
 *    %d - dest              %a - arg
 *    %f - function          %o - order (line #)
 *    %r - rva               %c - comment
 *
 *    %, - cond. comma       %# - cond. comment char 
 * Notes: like sprintf, this assumes 'buf' is large enough 
 */
	int j, count;
	char *d, *d2;
	char opbuf[128];
	char delim[] = ", ", delim2[] = "# ", none[] = "\0";
	struct name n = { 0 };
	struct comment cmt = { 0 };
	struct function f = {0};

	if (!buf || !format || !i)
		return (sys_set_lasterr(9010));

	
	buf[0] = 0;		//just in case
	d = d2 = none;

	for (j = 0; j < strlen(format); j++) {
		if (format[j] != '%') {
			strncat(buf, &format[j], 1);
		} else {
			j++;
			count = 0;	/* this is really only useful for comment */
			while (format[j] >= '0' && format[j] <= '9') {
				count *= 10;
				count += (format[j] - '0');
				j++;
			}
			memset( opbuf, 0, 128 );

			switch (format[j]) {
				case 'r':
					sprintf(opbuf, "%08X", i->rva);
					strncat( buf, opbuf, (count ? count : 8) );
					break;
				case 'm':
					intcode_getstr_mnem( i->opcode, opbuf, 128 );
					strncat( buf, opbuf, (count ? count : 32) );
					break;
				case 's':
					if ( i->opcode & EXT_ISA ) { /* directive */
						intcode_sprint_dirop( i, opbuf, 128 );
						strncat( buf, opbuf, (count ? count : 64) );
					} else if (i->sType) {
						strncat(buf, d, 2);	/* plunder comma */
						intcode_sprint_op( i->src, i->sType, 
									opbuf, 64 );
						strncat( buf, opbuf, (count ? count : 16) );
					} 
					d = none;			/* we used the comma */
					break;
				case 'd':
					if (i->dType && ! (i->opcode & EXT_ISA)) {
						strncat(buf, d, 2);	/* stomp comma */
						intcode_sprint_op( i->dest, i->dType, 
									opbuf, 64 );
					}
					strncat( buf, opbuf, (count ? count : 16) );
					d = none;			/* we ate the comma */
					break;
				case 'a':
					if (i->aType && ! (i->opcode & EXT_ISA)) {
						strncat(buf, d, 2);	/* chew up comma */
						intcode_sprint_op( i->arg, i->aType, 
									opbuf, 64 );
					}
					strncat( buf, opbuf, (count ? count : 16) );
					d = none;			/* we destroyed comma */
					break;
				case 'f':
					if ( bdb_index_find(FUNCTION_ID, &i->func, &f) ){
						if (bdb_index_find(NAME_RVA, &f.rva, &n)){
							strncat(buf, n.text, 
									(count ? count : 32));
						}
					}
					break;
				case 'o':
					sprintf( opbuf, "%d", i->order );
					strcat( buf, opbuf );
					break;
				case 'c':
					strncat(buf, d, 2);
					if (i->comment && bdb_index_find(COMMENT_ID, 
								&i->comment, &cmt)) {
						strncat(buf, d2, 2);
						strncat(buf, cmt.text,(count ? count : 64));
						d2 = none;
					} 
					break;
				case ',':
					/* insert a comma only if an operand follows it */
					d = delim;
					break;
				case '#':
					/* insert a comment_char only if cmt follows it */
					d2 = delim2;
					break;
				default:
					/* not a format metachar */
					strncat(buf, &format[j], 1);
					break;
				}
		}
	}
	return (1);
}
